/**
 * \file: mspin_demo_iap2_utilities.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * mySPIN Demo iAP2 Utilities
 *
 * \component: MSPIN
 *
 * \author: Bui Le Thuan / thuan.buile@vn.bosch.com
 *          Thilo Fickel ICT-ADITG/SW2 tfickel@de.adit-jv.com
 *
 * \copyright: (c) 2003 - 2013 ADIT Corporation
 *
 * \history
 * 0.1 TFickel Initial version
 *
 ***********************************************************************/

#ifndef MSPIN_IAP2_SUPPORT_DISABLED

#include "mspin_demo_iap2_utilities.h"
#include "mspin_logging.h"
#include "mspin_demo_iap2_connection.h"
#include <errno.h>
#include <sys/unistd.h>  //read/close
#include <sys/syscall.h> //syscall
#include <sys/poll.h>    //POLLIN/POLLOUT
#include <sys/utsname.h> //utsname
#include <pthread.h>
#include <sys/prctl.h>


BOOL Iap2Utilites::m_EndPoll = FALSE;

Iap2Utilites::Iap2Utilites()
{

}

Iap2Utilites::~Iap2Utilites()
{

}

/*Free Accessory Identification Values*/
void Iap2Utilites::freeAccInfo(iAP2AccessoryInfo_t *pAccInfo)
{
    U16 i;

    if (!pAccInfo )
    {
        return;
    }

    if (pAccInfo->iAP2AccessoryName != NULL)
    {
        free(pAccInfo->iAP2AccessoryName);
        pAccInfo->iAP2AccessoryName = NULL;
    }

    if (pAccInfo->iAP2AccessoryModelIdentifier != NULL)
    {
        free(pAccInfo->iAP2AccessoryModelIdentifier);
        pAccInfo->iAP2AccessoryModelIdentifier = NULL;
    }

    if (pAccInfo->iAP2AccessoryManufacturer != NULL)
    {
        free(pAccInfo->iAP2AccessoryManufacturer);
        pAccInfo->iAP2AccessoryManufacturer = NULL;
    }

    if (pAccInfo->iAP2AccessorySerialNumber != NULL)
    {
        free(pAccInfo->iAP2AccessorySerialNumber);
        pAccInfo->iAP2AccessorySerialNumber = NULL;
    }

    if (pAccInfo->iAP2AccessoryFirmwareVersion != NULL)
    {
        free(pAccInfo->iAP2AccessoryFirmwareVersion);
        pAccInfo->iAP2AccessoryFirmwareVersion = NULL;
    }

    if (pAccInfo->iAP2AccessoryHardwareVersion != NULL)
    {
        free(pAccInfo->iAP2AccessoryHardwareVersion);
        pAccInfo->iAP2AccessoryHardwareVersion = NULL;
    }

    if (pAccInfo->iAP2CommandsUsedByApplication != NULL)
    {
        free(pAccInfo->iAP2CommandsUsedByApplication);
        pAccInfo->iAP2CommandsUsedByApplication = NULL;
    }

    if (pAccInfo->iAP2CallbacksExpectedFromDevice != NULL)
    {
        free(pAccInfo->iAP2CallbacksExpectedFromDevice);
        pAccInfo->iAP2CallbacksExpectedFromDevice = NULL;
    }

    if (pAccInfo->iAP2PreferredAppBundleSeedIdentifier != NULL)
    {
        free(pAccInfo->iAP2PreferredAppBundleSeedIdentifier);
        pAccInfo->iAP2PreferredAppBundleSeedIdentifier = NULL;
    }

    if (pAccInfo->iAP2CurrentLanguage != NULL)
    {
        free(pAccInfo->iAP2CurrentLanguage);
        pAccInfo->iAP2CurrentLanguage = NULL;
    }

    if (pAccInfo->iAP2SupportedLanguage != NULL)
    {
        for (i=0; i<pAccInfo->iAP2SupportedLanguageCount; i++)
        {
            if (pAccInfo->iAP2SupportedLanguage[i] != NULL)
            {
                free(pAccInfo->iAP2SupportedLanguage[i]);
                pAccInfo->iAP2SupportedLanguage[i] = NULL;
            }
        }
        free(pAccInfo->iAP2SupportedLanguage);
        pAccInfo->iAP2SupportedLanguage = NULL;
    }

    if (pAccInfo->iAP2iOSAppInfo != NULL)
    {
        for (i=0; i<pAccInfo->iAP2SupportediOSAppCount; i++)
        {

                pAccInfo->iAP2iOSAppInfo[i].iAP2iOSAppIdentifier = 0;
                if (pAccInfo->iAP2iOSAppInfo[i].iAP2iOSAppName != NULL)
                {
                    free(pAccInfo->iAP2iOSAppInfo[i].iAP2iOSAppName);
                    pAccInfo->iAP2iOSAppInfo[i].iAP2iOSAppName = NULL;
                }
        }
        free(pAccInfo->iAP2iOSAppInfo);
        pAccInfo->iAP2iOSAppInfo = NULL;
        pAccInfo->iAP2SupportediOSAppCount = 0;
    }

    if (pAccInfo->iAP2USBDeviceSupportedAudioSampleRate != NULL)
    {
        free(pAccInfo->iAP2USBDeviceSupportedAudioSampleRate);
        pAccInfo->iAP2USBDeviceSupportedAudioSampleRate = NULL;
    }

    if (pAccInfo->iAP2BluetoothTransportMAC != NULL)
    {
        free(pAccInfo->iAP2BluetoothTransportMAC);
        pAccInfo->iAP2BluetoothTransportMAC = NULL;
    }

    //ToDo: free other parameters
}

void Iap2Utilites::freeAccConfig(iAP2AccessoryConfig_t* p_iAP2AccessoryConfig)
{
    if (p_iAP2AccessoryConfig->iAP2AuthDevicename != NULL)
    {
        free(p_iAP2AccessoryConfig->iAP2AuthDevicename);
        p_iAP2AccessoryConfig->iAP2AuthDevicename = NULL;
    }
    if (p_iAP2AccessoryConfig->iAP2AuthIoctlRegAddr != NULL)
    {
        free(p_iAP2AccessoryConfig->iAP2AuthIoctlRegAddr);
        p_iAP2AccessoryConfig->iAP2AuthIoctlRegAddr = NULL;
    }
    if (p_iAP2AccessoryConfig->iAP2AuthGPIOReset != NULL)
    {
        free(p_iAP2AccessoryConfig->iAP2AuthGPIOReset);
        p_iAP2AccessoryConfig->iAP2AuthGPIOReset = NULL;
    }
    if (p_iAP2AccessoryConfig->iAP2AuthGPIOReady != NULL)
    {
        free(p_iAP2AccessoryConfig->iAP2AuthGPIOReady);
        p_iAP2AccessoryConfig->iAP2AuthGPIOReady = NULL;
    }
}

S32 Iap2Utilites::iap2HdlPollMqEvent(S32 mqFD)
{
    S32 rc = IAP2_CTL_ERROR;
    char recvBuf[MYSPIN_MQ_MAX_SIZE + 1];

    /* receive message from AppThread */
    rc = iap2Receive(mqFD, &recvBuf[0], MYSPIN_MQ_MAX_SIZE + 1);
    if(rc > 0)
    {
        if (strncmp(&recvBuf[0], MYSPIN_MQ_CMD_STOP_POLL, strlen(MYSPIN_MQ_CMD_STOP_POLL)) == 0)
        {
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(mqFD=%d) leave poll thread", __FUNCTION__, mqFD);
            m_EndPoll = TRUE;
            rc = IAP2_OK;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(mqFD=%d) ERROR: buf='%s'", __FUNCTION__, mqFD, recvBuf);
            rc = IAP2_CTL_ERROR;
        }
    }
    else
    {
        rc = IAP2_CTL_ERROR;
    }

    return rc;
}

void Iap2Utilites::resetInitialParameter(iAP2InitParam_t* iAP2InitParam)
{
    /*TODO: Add more de-initialization functions to free the dynamic parts of the structures*/

    if (NULL != iAP2InitParam->p_iAP2AccessoryInfo)
    {
        freeAccInfo(iAP2InitParam->p_iAP2AccessoryInfo);
        free(iAP2InitParam->p_iAP2AccessoryInfo);
        iAP2InitParam->p_iAP2AccessoryInfo = NULL;
    }

    if(NULL != iAP2InitParam->p_iAP2AccessoryConfig)
    {
    	freeAccConfig(iAP2InitParam->p_iAP2AccessoryConfig);
        free(iAP2InitParam->p_iAP2AccessoryConfig);
        iAP2InitParam->p_iAP2AccessoryConfig = NULL;
    }
    if (NULL != iAP2InitParam->p_iAP2AccessoryConfig)
    {

        free(iAP2InitParam->p_iAP2AccessoryConfig);
        iAP2InitParam->p_iAP2AccessoryConfig = NULL;
    }

    if (NULL != iAP2InitParam->p_iAP2CSCallbacks)
    {
        free(iAP2InitParam->p_iAP2CSCallbacks);
        iAP2InitParam->p_iAP2CSCallbacks = NULL;
    }

    if (NULL != iAP2InitParam->p_iAP2FileTransferCallbacks)
    {
        free(iAP2InitParam->p_iAP2FileTransferCallbacks);
        iAP2InitParam->p_iAP2FileTransferCallbacks = NULL;
    }

    if (NULL != iAP2InitParam->p_iAP2EAPSessionCallbacks)
    {
        free(iAP2InitParam->p_iAP2EAPSessionCallbacks);
        iAP2InitParam->p_iAP2EAPSessionCallbacks = NULL;
    }

    if (NULL != iAP2InitParam->p_iAP2StackCallbacks)
    {
        free(iAP2InitParam->p_iAP2StackCallbacks);
        iAP2InitParam->p_iAP2StackCallbacks = NULL;
    }
}

S32 Iap2Utilites::createMsgQueue(mqd_t* mq_fd, const char* mq_name, S32 flag)
{
    S32 rc = IAP2_CTL_ERROR;
    mqd_t fd = -1;
    struct mq_attr attr;

    /* The only flag that can appear in this field is O_NONBLOCK. */
    attr.mq_flags = 0;
    /* Upper limit on the number of messages that may be placed on the queue using mq_send(). */
    attr.mq_maxmsg = 10;
    /* Upper limit on the size of messages that may be placed on the queue. */
    attr.mq_msgsize = MYSPIN_MQ_MAX_SIZE;
    /* Returns the number of messages currently held in the queue. */
    attr.mq_curmsgs = 0;

    /* create the message queue */
    fd = mq_open(mq_name, flag, 0644, &attr);
    if(fd != -1)
    {
        /* mq created */
        *mq_fd = fd;
        rc = IAP2_OK;
    }
    else
    {
        rc = IAP2_CTL_ERROR;
    }

    return rc;
}

void Iap2Utilites::iap2Sleep(U32 sleep_ms)
{
    S32 s32ReturnValue;
    struct timespec req;
    struct timespec remain;

    /* Initialize the structure */
    memset(&req, 0, sizeof(req));
    memset(&remain, 0, sizeof(remain));

    req.tv_sec = sleep_ms / 1000;
    req.tv_nsec = (sleep_ms % 1000) * 1000000;

    while (1)
    {
        s32ReturnValue = nanosleep(&req, &remain);

        if (s32ReturnValue == 0)
        {
            break;
        }
        else
        {
            if (errno == EINTR)
            {
                req.tv_sec = remain.tv_sec ;
                req.tv_nsec = remain.tv_nsec;
            }
            else
            {
                break;
            }
        }
    }
}


/* returns zero in case of success */
S32 Iap2Utilites::iap2Write(S32 fd, const char *pBuf, U32 size)
{
    S32 rc = IAP2_CTL_ERROR;
    char Buf[MYSPIN_MQ_MAX_SIZE];

    memset(Buf, 0, MYSPIN_MQ_MAX_SIZE);

    if (MYSPIN_MQ_MAX_SIZE >= size)
    {
        memcpy(&Buf[0], pBuf, size);

        /* send message to pollThread. return value of zero indicates success */
        rc = mq_send(fd, &Buf[0], MYSPIN_MQ_MAX_SIZE, 0);
        if (0 == rc)
        {
            mspin_log_printLn(eMspinVerbosityVerbose,
                    "%s(fd=%d, buf=%p, size=%d) send with rc=%d", __FUNCTION__, fd, pBuf, size, rc);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(fd=%d, buf=%p, size=%d) ERROR: Failed to send with rc=%d",
                    __FUNCTION__, fd, pBuf, size, rc);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(fd=%d, buf=%p, size=%d) ERROR: data to send is too big (>= %d)",
                __FUNCTION__, fd, pBuf, size, MYSPIN_MQ_MAX_SIZE);
        rc = IAP2_CTL_ERROR;
    }

    return rc;
}

S32 Iap2Utilites::addFDToPollFDs(iAP2PollFDs_t* getPollFDs, S32 numFDs, S32 fdToAdd, S16 eventToAdd)
{
    S32 rc = IAP2_OK;
    S32 i = 0;

    if(getPollFDs == NULL)
    {
        rc = IAP2_CTL_ERROR;
    }
    else
    {
        i = numFDs;

        getPollFDs[i].fd = fdToAdd;
        getPollFDs[i].event = eventToAdd;

        i++;
        rc = i;
    }

    return rc;
}

S32 Iap2Utilites::addFDsToFDset(iAP2PollFDs_t* getPollFDs, S32 numFDs, S32* maxfd, fd_set* to_readfds, fd_set* to_writefds)
{
    S32 rc = 0;
    S32 i = 0;

    if((getPollFDs == NULL) || (to_readfds == NULL)
        || (to_writefds == NULL) || (maxfd == NULL))
    {
        rc = -1;
    }
    else
    {
        /* adds the file descriptors to the fd_set */
        for(i = 0; i < numFDs; i++)
        {
            /* find highest-numbered file descriptor */
            if(getPollFDs[i].fd > *maxfd)
            {
                *maxfd = getPollFDs[i].fd;
            }

            if(getPollFDs[i].event == POLLIN)
            {
                /* FD_SET() adds the file descriptor to the fd_set */
                FD_SET(getPollFDs[i].fd, to_readfds);
                mspin_log_printLn(eMspinVerbosityVerbose,
                        "%s(...) fd=%d is set to read fds, event %d",
                        __FUNCTION__, getPollFDs[i].fd, getPollFDs[i].event);
            }
            else if(getPollFDs[i].event == POLLOUT)
            {
                /* FD_SET() adds the file descriptor to the fd_set */
                FD_SET(getPollFDs[i].fd, to_writefds);
                mspin_log_printLn(eMspinVerbosityVerbose,
                        "%s(...) fd=%d is set to write fds, event %d",
                        __FUNCTION__, getPollFDs[i].fd, getPollFDs[i].event);
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(...) ERROR: fd=%d is used for unknown event %d",
                        __FUNCTION__, getPollFDs[i].fd, getPollFDs[i].event);
            }
        }
        rc = 0;
    }

    return rc;
}

/* returns zero in case of success */
S32 Iap2Utilites::iap2Receive(S32 fd, char *pRecvBuf, U32 size)
{
    S32 rc = IAP2_CTL_ERROR;
    char buf[MYSPIN_MQ_MAX_SIZE + 1];
    S32 bytesReceived = 0;

    /* On success, mq_receive() return the number of bytes in the received message */
    bytesReceived = mq_receive(fd, buf, MYSPIN_MQ_MAX_SIZE, NULL);
    if( (bytesReceived >= 0) && ((U32)bytesReceived < size) )
    {
        buf[bytesReceived] = '\0';
        memcpy(pRecvBuf, &buf[0], bytesReceived+1);
        mspin_log_printLn(eMspinVerbosityVerbose,
                "%s(fd=%d, ...) received %d bytes",
                __FUNCTION__, fd, bytesReceived);
        rc = bytesReceived+1;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(fd=%d, ...) ERROR: received %d bytes",
                __FUNCTION__, fd, bytesReceived);
        rc = IAP2_CTL_ERROR;
    }

    return rc;
}

void* Iap2Utilites::iap2Poll(void* exinf)
{
    iAP2Device_t *p_iAP2Device = (iAP2Device_t*)exinf;
    S32 rc = IAP2_CTL_ERROR;
    mqd_t mq_fd = -1;
    iAP2GetPollFDs_t getPollFDs;
    iAP2PollFDs_t pollFDs[10];
    S32 cntfds = 0;
    S32 j;
    S32 nfds = 0; // highest-numbered file descriptor in any of the three sets, plus 1
    fd_set read_fds; // will be watched to see if characters become available for reading
    fd_set write_fds;

    prctl(PR_SET_NAME,"iap2Poll",0,0,0);

    if (!p_iAP2Device)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(%p) FATAL ERROR: p_iAP2Device is NULL",
                __FUNCTION__, exinf);
        return NULL;
    }

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(%p) thread started", __FUNCTION__, exinf);

    //Create message queue
    rc = createMsgQueue(&mq_fd, MYSPIN_MQ_NAME, O_RDONLY);
    if (IAP2_OK != rc)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(%p) ERROR: Failed to create message queue with code=%d",
                __FUNCTION__, exinf, rc);
        return NULL;
    }

    //Get file descriptor
    rc = iAP2GetPollFDs(p_iAP2Device, &getPollFDs);
    for (j=0; j < getPollFDs.numberFDs; j++)
    {
        rc = addFDToPollFDs(&pollFDs[0], j, getPollFDs.fds[j].fd, getPollFDs.fds[j].event);
        if(rc >= IAP2_OK)
        {
            cntfds = rc;
            rc = IAP2_OK;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(%p) iap2AddFDToPollFDs=%d %d added",
                    __FUNCTION__, exinf, rc, getPollFDs.fds[j].event);
        }
    }

    if (cntfds == getPollFDs.numberFDs)
    {
        rc = addFDToPollFDs(&pollFDs[0], cntfds, mq_fd, POLLIN);
        if(rc >= IAP2_OK)
        {
            cntfds = rc;
            rc = IAP2_OK;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(%p) iap2AddFDToPollFDs=%d POLLIN added", __FUNCTION__, exinf, rc);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(%p) ERROR: cntfds: %d != %d numberFDs",
                __FUNCTION__, exinf, cntfds, getPollFDs.numberFDs);
        rc = IAP2_CTL_ERROR;
        return NULL;
    }

    //Main loop
    while (FALSE == m_EndPoll)
    {
        /* FD_ZERO() clears out the fd_set, so it doesn't contain any file descriptors */
        FD_ZERO(&read_fds);
        FD_ZERO(&write_fds);

        rc = addFDsToFDset(&pollFDs[0], cntfds, &nfds, &read_fds, &write_fds);
        if (rc != IAP2_OK)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(%p) ERROR: Failed to call iap2AddFDsToFDset with rc=%d",
                    __FUNCTION__, exinf, rc);
        }

        rc = select(nfds+1, &read_fds, &write_fds, NULL, NULL);
        if (rc > 0)
        {
            for (j = 0; j < cntfds; j++)
            {
                if ((j < cntfds) && (FD_ISSET(pollFDs[j].fd, &read_fds)))
                {
                    mspin_log_printLn(eMspinVerbosityVerbose,
                            "%s(%p) read fd[%d] %d (event: %d) is set",
                            __FUNCTION__, exinf, j, pollFDs[j].fd, pollFDs[j].event);

                    if (mq_fd == pollFDs[j].fd)
                    {
                        rc = iap2HdlPollMqEvent(pollFDs[j].fd);
                    }
                    else
                    {
                        rc = iAP2HandleEvent(p_iAP2Device, pollFDs[j].fd, pollFDs[j].event);
                    }
                }
                if( (j < cntfds) && (FD_ISSET(pollFDs[j].fd, &write_fds)) )
                {
                    mspin_log_printLn(eMspinVerbosityVerbose,
                            "%s(%p) write fd[%d] %d (event: %d) is set",
                            __FUNCTION__, exinf, j, pollFDs[j].fd, pollFDs[j].event);

                    rc = iAP2HandleEvent(p_iAP2Device, pollFDs[j].fd, pollFDs[j].event);
                }

                if (IAP2_DEV_NOT_CONNECTED == rc)
                {
                    mspin_log_printLn(eMspinVerbosityError,
                           "%s(%p) ERROR: device disconnected",
                           __FUNCTION__, exinf);
                    m_EndPoll = TRUE;

                    break;
                }
            }
        }
        else
        {
            /* rc = 0 : select() timed out
             * rc < 0 : indicates an error (check errno)
             */

            if (rc < 0)
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(%p) ERROR: myPollFDs failed = %d (errno='%s')",
                        __FUNCTION__, exinf, rc, strerror(errno));
                m_EndPoll = TRUE;
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(%p) ERROR: myPollFDs failed = %d (timeout)",
                        __FUNCTION__, exinf, rc);
            }
        }
    }

    if (mq_fd > 0)
    {
        rc = mq_close(mq_fd);
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(%p) exit", __FUNCTION__, exinf);

    pthread_exit((void*)exinf);
    return NULL;
}

void Iap2Utilites::iap2StopPolling(BOOL endPoll)
{
    m_EndPoll = endPoll;
}

void Iap2Utilites::parseIdRejectedMsg(iAP2IdentificationRejectedParameter* idParameter)
{
    U16 i = 0;

    if (idParameter)
    {
        //Firmware version
        if (idParameter->iAP2AccessoryFirmwareVersion_count> 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: firmware version count is %d",
                    __FUNCTION__, idParameter->iAP2AccessoryFirmwareVersion_count);
            for (i = 0; i < idParameter->iAP2AccessoryFirmwareVersion_count; i++)
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s() ERROR: %d firmware version is %s",
                        __FUNCTION__,  i,
                        (0 == idParameter->iAP2AccessoryFirmwareVersion[i]) ? "false" : "true");
            }
        }

        //Hardware version
        if (idParameter->iAP2AccessoryHardwareVersion_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: hardware version count is %d",
                    __FUNCTION__, idParameter->iAP2AccessoryHardwareVersion_count);

            for (i = 0; i < idParameter->iAP2AccessoryHardwareVersion_count; i++)
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s() ERROR: %d firmware version is %s",
                        __FUNCTION__, i,
                        (0 == idParameter->iAP2AccessoryHardwareVersion[i]) ? "false" : "true");
            }
        }

        //Manufacturer
        if (idParameter->iAP2AccessoryManufacturer_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: manufacturer count is %d",
                    __FUNCTION__, idParameter->iAP2AccessoryManufacturer_count);
        }

        //Model identifier
        if (idParameter->iAP2AccessoryModelIdentifier_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: model identifier count is %d",
                    __FUNCTION__, idParameter->iAP2AccessoryModelIdentifier_count);
        }

        //Accessory name
        if (idParameter->iAP2AccessoryHardwareVersion_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: accessory hardware version count is %d",
                    __FUNCTION__, idParameter->iAP2AccessoryName_count);
            for (i = 0; i < idParameter->iAP2AccessoryHardwareVersion_count; i++)
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s() ERROR: %d accessory name is %s",
                        __FUNCTION__, i,
                        (0 == idParameter->iAP2AccessoryName[i]) ? "false" : "true");
            }
        }

        //Accessory serial number
        if (idParameter->iAP2AccessorySerialNumber_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: accessory serial number count is %d",
                    __FUNCTION__, idParameter->iAP2AccessorySerialNumber_count);
        }

        //Bluetooth transport component
        if (idParameter->iAP2BluetoothTransportComponent_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: bluetooth transport component count is %d",
                    __FUNCTION__, idParameter->iAP2BluetoothTransportComponent_count);
        }

        //Current language
        if (idParameter->iAP2CurrentLanguage_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: current language count is %d",
                    __FUNCTION__, idParameter->iAP2CurrentLanguage_count);
        }

        //HID component
        if (idParameter->iAP2iAP2HIDComponent_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: HID component count is %d",
                    __FUNCTION__, idParameter->iAP2iAP2HIDComponent_count);
        }

        //Location information component
        if (idParameter->iAP2LocationInformationComponent_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: location information component count is %d",
                    __FUNCTION__, idParameter->iAP2LocationInformationComponent_count);
        }

        //Location information component
        if (idParameter->iAP2MaximumCurrentDrawnFromDevice_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s() ERROR: maximum current drawn from device count is %d",
                    __FUNCTION__, idParameter->iAP2MaximumCurrentDrawnFromDevice_count);
        }

        //Message received from device
        if (idParameter->iAP2MessagesRecievedfromDevice_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: messages received from device count is %d",
                    __FUNCTION__, idParameter->iAP2MessagesRecievedfromDevice_count);
        }

        //Messages sent by accessory
        if (idParameter->iAP2MessagesSentByAccessory_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: messages sent by accessory count is %d",
                    __FUNCTION__, idParameter->iAP2MessagesSentByAccessory_count);
        }

        //Power source type
        if (idParameter->iAP2PowerSourceType_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: power source type count is %d",
                    __FUNCTION__, idParameter->iAP2PowerSourceType_count);
        }

        //Preferred app bundle seed identifier
        if (idParameter->iAP2PreferredAppBundleSeedIdentifier_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: preferred app bundle seed identifier count is %d",
                    __FUNCTION__, idParameter->iAP2PreferredAppBundleSeedIdentifier_count);
        }

        //Serial transport component
        if (idParameter->iAP2SerialTransportComponent_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: serial transport component count is %d",
                    __FUNCTION__, idParameter->iAP2SerialTransportComponent_count);
        }

        //Supported External Accessory protocol
        if (idParameter->iAP2SupportedExternalAccessoryProtocol_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s() ERROR: Supported External Accessory protocol count is %d",
                    __FUNCTION__, idParameter->iAP2SupportedExternalAccessoryProtocol_count);
        }

        //Supported language
        if (idParameter->iAP2SupportedLanguage_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: Supported language count is %d",
                    __FUNCTION__, idParameter->iAP2SupportedLanguage_count);
        }

        //USB device transport component
        if (idParameter->iAP2USBDeviceTransportComponent_count)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: USB device transport component count is %d",
                    __FUNCTION__, idParameter->iAP2USBDeviceTransportComponent_count);
        }

        //USB host HID component
        if (idParameter->iAP2USBHostHIDComponent_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: USB host HID component count is %d",
                    __FUNCTION__, idParameter->iAP2USBHostHIDComponent_count);
        }

        //USB host transport component
        if (idParameter->iAP2USBHostTransportComponent_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: USB host transport component count is %d",
                    __FUNCTION__, idParameter->iAP2USBHostTransportComponent_count);
        }

        //Vehicle information component
        if (idParameter->iAP2VehicleInformationComponent_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: Vehicle information component is %d",
                    __FUNCTION__, idParameter->iAP2VehicleInformationComponent_count);
        }

        //Vehicle status component
        if (idParameter->iAP2VehicleStatusComponent_count > 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: Vehicle status component is %d",
                    __FUNCTION__, idParameter->iAP2VehicleStatusComponent_count);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s() ERRROR: idParameter is NULL", __FUNCTION__);
    }
}

S32 Iap2Utilites::iap2CopyString(U8* pString, U8** ppDestination)
{
    S32 StringLength = strnlen((const char*)pString, STRING_MAX);
    *ppDestination = (U8*)calloc(1, (StringLength + IAP2_NULL_CHAR_LEN));
    if (*ppDestination != NULL)
    {
        memcpy(*ppDestination, pString, (StringLength + IAP2_NULL_CHAR_LEN) );
        (*ppDestination)[StringLength] = '\0';
        return IAP2_OK;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s('%s') FATAL ERROR: Could not allocate memory for string",
                __FUNCTION__, pString);
        return IAP2_ERR_NO_MEM;
    }
}

//Load_module - load kernel module
S32 Iap2Utilites::loadModule(iAP2InitParam_t* iap2InitParam, const char* name, int length)
{
    int fd, rc;
    char path_name[256];
    char buf[1024*1024];

    //Determine the path of the module
    struct utsname system_name;

    if (uname(&system_name) == 0)
    {
        sprintf(path_name, MYSPIN_DEMO_IAP2_LINUX_USB_GADGET_MODULE_PATH, system_name.release, name);

        fd = open(path_name, O_RDONLY);
        if (fd > 0)
        {
            rc = read(fd, buf, sizeof(buf));
            close(fd);

            if (0 == strncmp(name, MYSPIN_DEMO_IAP2_LIBCOMPOSITE_MODULE_NAME, length))
            {
                //Issue the init_module system call
                return syscall(__NR_init_module, buf, rc, "");
            }
            else if(0 == strncmp(name, MYSPIN_DEMO_IAP2_GADGET_FFS_MODULE_NAME, length))
            {
                if (!iap2InitParam->p_iAP2AccessoryInfo)
                {
                    mspin_log_printLn(eMspinVerbosityError,
                            "%s(iap2InitParam=%p, name='%s', len=%d) ERROR: iap2InitParam->p_iAP2AccessoryInfo is NULL",
                            __FUNCTION__, iap2InitParam, name, length);
                    return -1;
                }

                char sysCallStr[256];
                memset(&sysCallStr[0], 0, 256);
                sprintf(&sysCallStr[0],
                        "idVendor=%s idProduct=%s iManufacturer=%s iProduct=%s iSerialNumber=%s bcdDevice=%s",
                        iap2InitParam->p_iAP2AccessoryInfo->iAP2AccessoryVendorId, \
                        iap2InitParam->p_iAP2AccessoryInfo->iAP2AccessoryProductId, \
                        iap2InitParam->p_iAP2AccessoryInfo->iAP2AccessoryManufacturer, \
                        iap2InitParam->p_iAP2AccessoryInfo->iAP2AccessoryName, \
                        iap2InitParam->p_iAP2AccessoryInfo->iAP2AccessorySerialNumber, \
                        iap2InitParam->p_iAP2AccessoryInfo->iAP2AccessoryBcdDevice);

                //Issue the init_module system call
                return syscall(__NR_init_module, buf, rc, &sysCallStr[0]);
            }
            else
            {
                //Error. Unknown module name
            }
        }
    }

    return -1;
}

//unload_module - unload kernel module
S32 Iap2Utilites::unloadModule(const char* name)
{
    //issue the delete_module system call
    return syscall(__NR_delete_module, name, 0);
}

#endif //#ifndef MSPIN_IAP2_SUPPORT_DISABLED
